//------------------------------------------------------------------------------
// Example software to demonstrate the usage of the HDQ communication library.
// The temperature from an bq27541 gas-gauge IC that is connected via HDQ is
// read out periodically and temperature changes are indicated by operating
// the GPIO pin of the bq27541 device. See application report for more details.
//
// R. Wu
// Texas Instruments Inc.
// February 2010
// Built with IAR Embedded Workbench Version: 4.21A
//------------------------------------------------------------------------------
#include <stdlib.h>
#include "bq27541.h"
#include "..\..\lib\VLO_RAND\vlo_rand.h"
#include "..\..\lib\SHA1_HMAC\SHA1_HMAC.h"
#include "..\..\lib\HDQ\HDQ.h"

//------------------------------------------------------------------------------

#define ATRATE_MA            -100           // USER CONFIG: AtRate setting (mA)
#define BUFFERSIZE             32           // # of bytes for Tx & Rx buffers

extern UINT8 Message[RANDMESGNUMBYTES]; // random message sent to the bq device
extern UINT8 Key[SECRETKEYNUMBYTES]; // secret key - should match contents of bq 
extern UINT32 Digest_32[5]; // Result of SHA1/HMAC obtained by MCU stored here
extern UINT32 H[5];

UINT8 DeviceID[DEVICEIDNUMBYTES];           // Stores the Device ID data
UINT8 Digest[DIGESTNUMBYTES];               // SHA1 response from the bq27541

unsigned char TxData[BUFFERSIZE];           // Stores data bytes to be TX'd
unsigned char RxData[BUFFERSIZE];           // Stores data bytes that are RX'd
unsigned int  temperature;                  // Stores temperature
unsigned int  voltage;                      // Stores voltage
  signed int  atrate;                       // Stores AtRate
unsigned int  artte;                        // Stores AtRate Time to Empty
unsigned int  soc;                          // Stores State of Charge
unsigned int  dcap;                         // Stores Design Capacity
unsigned int  dnamelen;                     // Stores Device Name Length

void delay(void)
{
  __delay_cycles(500000);                   // Delay  
}

unsigned int transBytes2Int(unsigned char msb, unsigned char lsb)
{
  unsigned int tmp;
  
  tmp = ((msb << 8) & 0xFF00);
  return ((unsigned int)(tmp + lsb) & 0x0000FFFF);  
}

void MSP430_bq27541_error(void)
{
  while (1)                                 // Loop forever
  {
    P1OUT ^= BIT0;                          // Toggle LED
    delay();                                // Delay
  }
}

void main(void)
{
  unsigned int i;
  unsigned int sum = 0;
  unsigned char checksum;
  unsigned char read;
  unsigned char msb, lsb;

  WDTCTL = (WDTPW | WDTTMSEL);              // WDT as interval timer
  P1OUT &= ~BIT0;
  P1SEL &= ~BIT0;                           // P1.x GPIO function
  P1DIR |=  BIT0;                           // P1.x output direction

  HDQSetup();                               // Do basic HDQ setup
 
  __enable_interrupt();

  // Read Temperature (units = 0.1K)
  RxData[1] = HDQRead(bq27541CMD_TEMP_MSB);
  RxData[0] = HDQRead(bq27541CMD_TEMP_LSB);  
  temperature = transBytes2Int(RxData[1], RxData[0]);
    
  // Read Voltage (units = mV)
  RxData[1] = HDQRead(bq27541CMD_VOLT_MSB);
  RxData[0] = HDQRead(bq27541CMD_VOLT_LSB);  
  voltage = transBytes2Int(RxData[1], RxData[0]);
  
  // Set AtRate (units = mA)
  atrate = ATRATE_MA;
  msb = ((atrate >> 8) & 0x00FF);
  lsb = (atrate & 0x00FF);
  HDQWrite(bq27541CMD_AR_LSB, lsb);         // Must write the LSB before MSB !!!
  HDQWrite(bq27541CMD_AR_MSB, msb);
  
  // Verify AtRate was set correctly (units = mA)
  RxData[0] = HDQRead(bq27541CMD_AR_MSB);
  RxData[1] = HDQRead(bq27541CMD_AR_LSB); 
  if ((RxData[0] != msb) || (RxData[1] != lsb))
  {
    MSP430_bq27541_error();                 // Signal error condition occurred
  }
  
  // Read AtRate Time to Empty (units = Minutes)
  RxData[1] = HDQRead(bq27541CMD_ARTTE_MSB);
  RxData[0] = HDQRead(bq27541CMD_ARTTE_LSB); 
  artte = transBytes2Int(RxData[1], RxData[0]);
  
  // Read State of Charge (units = %)
  RxData[1] = HDQRead(bq27541CMD_SOC_MSB);
  RxData[0] = HDQRead(bq27541CMD_SOC_LSB); 
  soc = transBytes2Int(RxData[1], RxData[0]);

  // Read Design Capacity (units = mAH)
  RxData[1] = HDQRead(bq27541CMD_DCAP_MSB);
  RxData[0] = HDQRead(bq27541CMD_DCAP_LSB);
  dcap = transBytes2Int(RxData[1], RxData[0]);
  
  // Read Device Name Length
  dnamelen = HDQRead(bq27541CMD_DNAMELEN);
  
  // Read Device Name (Rx buffer should end up with ASCII chars for 'bq27541')
  for (i = 0; i < 7; i++)
  {
    RxData[i] = HDQRead(bq27541CMD_DNAME + i);
  }
  
  // Write & read back 32 bytes of data in Manufacturer Info Block A
  for (i = 0; i < BUFFERSIZE; i++)          
  {
    TxData[i] = i;                          // Initialize data to be written
  }
  HDQWrite(bq27541CMD_DFDCNTL, 0);          // BlockDataControl() = 0x00
  HDQWrite(bq27541CMD_DFCLS, 58);           // Write the subclass value
  HDQWrite(bq27541CMD_DFBLK, 0);            // Select offset within the flash  
  for (i = 0; i < BUFFERSIZE; i++)          // Compute the checksum of the block
  {
    sum += TxData[i];                       // Calculate the sum of the values  
  }
  checksum = (0xFF - (sum & 0x00FF));       // Compute checksum based on the sum
  HDQWrite(bq27541CMD_DFDCKS, checksum);    // Write the checksum value
  for (i = 0; i < BUFFERSIZE; i++)          // Write 32 bytes to Info Block A
  {
    HDQWrite((bq27541CMD_ADF+i), TxData[i]);
  }  
  
  for (i = 0; i < BUFFERSIZE; i++)          // Check if writes were successful
  {
    RxData[i] = HDQRead(bq27541CMD_ADF+i);  // Read the contents of the block
    if (TxData[i] != RxData[i])             // Tx & Rx data values match?
    {
      MSP430_bq27541_error();               // Signal error condition occurred
    }
  }
  
  // TODO: Insert the private 128-bit key that is stored in the bq27541
  // Key[15..8] = K1 (highest 64 bits of the key)
  // Key[ 7..0] = K0 (lowest 64 bits of the key)
  // In this example 0x0123456789ABCDEFFEDCBA9876543210 is used since a fresh
  // unprogrammed bq27541 device has this as its default for the 128-bit key.
  Key[15] = 0x01;
  Key[14] = 0x23;
  Key[13] = 0x45;
  Key[12] = 0x67;
  Key[11] = 0x89;
  Key[10] = 0xAB;
  Key[ 9] = 0xCD;
  Key[ 8] = 0xEF;
  Key[ 7] = 0xFE;
  Key[ 6] = 0xDC;
  Key[ 5] = 0xBA;
  Key[ 4] = 0x98;
  Key[ 3] = 0x76;
  Key[ 2] = 0x54;
  Key[ 1] = 0x32;
  Key[ 0] = 0x10;
  
  // Perform my own SHA1 (Host side)
  read = (UINT8)TI_getRandomIntegerFromVLO();// Use instability of VLO for rand
  srand(read);                              // Plant seed based on random value 
  for (i = 0; i < RANDMESGNUMBYTES; i++)    // Initialize random challenge bytes
  {
    Message[i] = rand();                    // Generate 1 random challenge byte
  }
  SHA1_authenticate();                      // Execute SHA-1/HMAC algorithm
  
  // Authenticate the bq27541
  HDQWrite(bq27541CMD_DFDCNTL, 1);// BlockDataControl() = 0x01
  // Write 20-byte authentication challenge to bq27541 starting at 0x40
  for (i = 0; i < RANDMESGNUMBYTES; i++)
  {
    HDQWrite(bq27541CMD_ADF+i, Message[i]);  
  }
  // Write checksum for the challenge to the bq27541
  sum = 0;
  for (i = 0; i < RANDMESGNUMBYTES; i++)    // Compute the checksum of the block
  {
    sum += Message[i];                      // Calculate the sum of the values  
  }
  checksum = (0xFF - (sum & 0x00FF));       // Compute checksum based on the sum
  HDQWrite(bq27541CMD_ACKSDFD, checksum);
  // Read back the digest from the bq27541
  for (i = 0; i < RANDMESGNUMBYTES; i++)
  {
    RxData[i] = HDQRead(bq27541CMD_ADF+i);
  }
  
  // The 20 bytes of the digest returned by the bq27541 is arranged in 32-bit
  // words so that it can be compared with the results computed by the MCU
  Digest_32[4] = (UINT32)(RxData[ 0])*0x00000001 +
                 (UINT32)(RxData[ 1])*0x00000100 +
                 (UINT32)(RxData[ 2])*0x00010000 +
                 (UINT32)(RxData[ 3])*0x01000000;
  Digest_32[3] = (UINT32)(RxData[ 4])*0x00000001 +
                 (UINT32)(RxData[ 5])*0x00000100 +
                 (UINT32)(RxData[ 6])*0x00010000 +
                 (UINT32)(RxData[ 7])*0x01000000; 
  Digest_32[2] = (UINT32)(RxData[ 8])*0x00000001 +
                 (UINT32)(RxData[ 9])*0x00000100 +
                 (UINT32)(RxData[10])*0x00010000 +
                 (UINT32)(RxData[11])*0x01000000;
  Digest_32[1] = (UINT32)(RxData[12])*0x00000001 +
                 (UINT32)(RxData[13])*0x00000100 +
                 (UINT32)(RxData[14])*0x00010000 +
                 (UINT32)(RxData[15])*0x01000000;
  Digest_32[0] = (UINT32)(RxData[16])*0x00000001 +
                 (UINT32)(RxData[17])*0x00000100 +
                 (UINT32)(RxData[18])*0x00010000 +
                 (UINT32)(RxData[19])*0x01000000;

  // The results produced by the MCU and bq27541 must match for success
  if ( (Digest_32[0] == H[0]) && (Digest_32[1] == H[1]) &&
       (Digest_32[2] == H[2]) && (Digest_32[3] == H[3]) &&
       (Digest_32[4] == H[4]) )
  {
    // Set P1.0 LED on MSP430 EVM to signal that command sequence was successful
    P1OUT |= BIT0;
  }
  else
  {
    MSP430_bq27541_error();                 // Error condition
  }
  
}
